/** @file   youko.cpp
 * @brief   Implementation of Youko - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "youko.h"
#include "youkocontroller.h"
#include "settings.h"
#include "objectmessage.h"
#include "datatypes.h"
#include "soundsamples.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
Youko::Youko() :
  NpcCharacter()
{
  int oid = ObjectID::TYPE_YOUKO;
  this->setArmor( Settings::floatObjProp( oid, "armor:" ) );
  this->boundingSphere( Settings::floatObjProp( oid, "bsphere:" ) );
  m_movementVec.vy = - Settings::floatObjProp( oid, "walk_spd:" );
  
  // Init the controller.
  YoukoController* pC = new YoukoController( this );
  this->setController( pC );
  
  int animID = GameAnims::EIdle;
  const Animation& anim = GameAnims::findAnimation( AnimId::KYouko, animID );
  this->setAnimation( anim, animID );
  
  // Setup the sounds
  m_soundIndices.at( GameObject::SND_ATTACK ) = SMP_WEAPON1;
}



/** Destructor
 */
Youko::~Youko()
{
  /** The controller object will be deleted in GameObject's destructor.
   */
}




///
/// Public methods
/// ==============

/** Updates the Youko
 */
void Youko::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->state( GameObject::STATE_KILLED );
      this->changeAnimation( GameAnims::EKilled );
    }
    return;
  }
  
  YoukoController* pC = dynamic_cast<YoukoController*>( this->getController() );
  pC->update();
  this->changeAngle( pC->turn() );
  
  if ( pC->forward() == 0 ) {
    this->changeAnimation( GameAnims::EIdle );
    
  } else if ( pC->forward() == 1 ) {
    Vec2D mvec( m_movementVec );
    mvec.rotate( this->angle() );
    this->move( mvec );
    this->changeAnimation( GameAnims::EWalk );
  }
  
  if ( pC->shoot() != 0 && NpcCharacter::m_weapon.id() != Weapon::W_NOWEAPON ) {
    this->changeAnimation( GameAnims::EShoot );
    if ( this->reloading() == false ) {
      this->handleShooting();
    }
  }
}



/** Handles the messages
 */
void Youko::messagePort( const ObjectMessage& rMessage )
{
  if ( rMessage.m_id == ObjectMessage::KResponse ) {
    return;
  }
  
  // Create a response if possible.
  if ( rMessage.m_pSender != 0 ) {
    ObjectMessage tmpMessage = ObjectMessage::createResponse( rMessage );
    rMessage.m_pSender->messagePort( tmpMessage );
  }
  
  // Analyze the message
  switch ( rMessage.m_id ) {
    case ( ObjectMessage::KControllerState ): {
      YoukoController* pC = dynamic_cast<YoukoController*>(
                            this->getController() );
      int newState = rMessage.m_params.getInt( "message_data:" );
      pC->youkoState( newState );
      break;
    }
    case ( ObjectMessage::KWeaponChange ): {
      this->setWeapon( Weapon( rMessage.m_params.getInt( "message_data:" ) ) );
      break;
    }
  }
}



/** Sets the weapon
 */
void Youko::setWeapon( Weapon w )
{
  NpcCharacter::setWeapon( w );
  switch ( w.id() ) {
    case ( Weapon::W_NOWEAPON ): {
      NpcCharacter::m_animationCombine = GameAnims::getAnimCombine( AnimId::KYouko );
      const Animation& anim =
          GameAnims::findAnimation( AnimId::KYouko, GameAnims::EIdle );
      this->setAnimation( anim, GameAnims::EIdle );
      // Setup the sounds
      m_soundIndices.at( GameObject::SND_ATTACK ) = -1;
      break;
    }
    case ( Weapon::W_RIFLE ): {
      NpcCharacter::m_animationCombine = GameAnims::getAnimCombine( AnimId::KArmedYouko );
      const Animation& anim =
          GameAnims::findAnimation( AnimId::KArmedYouko, GameAnims::EIdle );
      this->setAnimation( anim, GameAnims::EIdle );
      // Setup the sounds
      m_soundIndices.at( GameObject::SND_ATTACK ) = SMP_WEAPON1;
      break;
    }
    case ( Weapon::W_SHOTGUN ): {
      NpcCharacter::m_animationCombine = GameAnims::getAnimCombine( AnimId::KArmedYouko );
      const Animation& anim =
          GameAnims::findAnimation( AnimId::KArmedYouko, GameAnims::EIdle );
      this->setAnimation( anim, GameAnims::EIdle );
      // Setup the sounds
      m_soundIndices.at( GameObject::SND_ATTACK ) = SMP_WEAPON2;
      break;
    }
  }
}



/** Kills the Youko
 */
void Youko::kill()
{
  if ( Settings::cheatYouko == true ) {
    this->setHealth( 100 );
    return;
  }
  NpcCharacter::kill();
}



///
/// Getter methods
/// ==============

/** Returns the type of us.
 */
ObjectID::Type Youko::objectType() const
{
  return ObjectID::TYPE_YOUKO;
}




///
/// Private or Protected methods
/// ============================

} // end of namespace
